<?php
session_start();
include '../config.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

// Obtener el mes y año seleccionados o usar el actual
$selected_month = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('m');
$selected_year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');

// Validar mes y año
if ($selected_month < 1 || $selected_month > 12) $selected_month = (int)date('m');
if ($selected_year < 2020 || $selected_year > (int)date('Y')) $selected_year = (int)date('Y');

// Obtener el primer y último día del mes seleccionado
$first_day = sprintf('%04d-%02d-01', $selected_year, $selected_month);
$last_day = date('Y-m-t', strtotime($first_day));
$days_in_month = date('t', strtotime($first_day));

// Nombre del mes en español
$meses = array(
    "01" => "Enero", "02" => "Febrero", "03" => "Marzo", "04" => "Abril",
    "05" => "Mayo", "06" => "Junio", "07" => "Julio", "08" => "Agosto",
    "09" => "Septiembre", "10" => "Octubre", "11" => "Noviembre", "12" => "Diciembre"
);
$nombre_mes = $meses[sprintf("%02d", $selected_month)];

// Crear un array con todos los días del mes
$all_dates = array();
$all_counts = array();
for ($i = 1; $i <= $days_in_month; $i++) {
    $date = sprintf('%04d-%02d-%02d', $selected_year, $selected_month, $i);
    $all_dates[] = date('d/m', strtotime($date));
    $all_counts[$date] = 0;
}

// Obtener las vistas reales para el mes seleccionado
$query = "SELECT 
            DATE(view_date) as date,
            COUNT(*) as total
          FROM video_views 
          WHERE view_date BETWEEN '$first_day' AND '$last_day 23:59:59'
          GROUP BY DATE(view_date)";
$result = mysqli_query($conn, $query);

// Rellenar los conteos reales
while ($row = mysqli_fetch_assoc($result)) {
    $all_counts[$row['date']] = (int)$row['total'];
}

// Convertir a array simple para el gráfico
$counts = array_values($all_counts);

// Obtener estadísticas totales con consultas optimizadas
$stats_query = "SELECT 
                (SELECT COUNT(*) FROM videos) as total_videos,
                (SELECT COUNT(*) FROM video_views) as total_views,
                (SELECT COUNT(*) FROM video_views 
                 WHERE DATE(view_date) = CURDATE()) as today_views,
                (SELECT COUNT(*) FROM video_views 
                 WHERE YEAR(view_date) = $selected_year 
                 AND MONTH(view_date) = $selected_month) as month_views";
$stats_result = mysqli_query($conn, $stats_query);
$stats = mysqli_fetch_assoc($stats_result);

// Obtener el código del contador
$settings_query = "SELECT counter_code FROM settings WHERE id = 1";
$settings_result = mysqli_query($conn, $settings_query);
$settings = mysqli_fetch_assoc($settings_result);

// Obtener lista de años disponibles
$years_query = "SELECT DISTINCT YEAR(view_date) as year 
                FROM video_views 
                ORDER BY year DESC";
$years_result = mysqli_query($conn, $years_query);
$available_years = [];
while ($year = mysqli_fetch_assoc($years_result)) {
    $available_years[] = $year['year'];
}
if (empty($available_years)) {
    $available_years[] = date('Y');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Free and Simple Video Hosting</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
    :root {
        --primary-gradient: linear-gradient(135deg, #6e8efb, #a777e3);
        --secondary-gradient: linear-gradient(135deg, #a777e3, #6e8efb);
        --card-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
        --transition: all 0.3s ease;
    }

    body {
        background-color: #f8f9fe;
        font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
    }

    .dashboard-header {
        background: var(--primary-gradient);
        padding: 2rem 0;
        margin-bottom: 2rem;
        color: white;
        border-radius: 0 0 25px 25px;
        box-shadow: var(--card-shadow);
    }

    .upload-section {
        background: white;
        border-radius: 20px;
        box-shadow: var(--card-shadow);
        padding: 2rem;
        margin-bottom: 2rem;
        position: relative;
        overflow: hidden;
    }

    .upload-section::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: var(--primary-gradient);
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        border-radius: 20px;
        padding: 1.5rem;
        text-align: center;
        box-shadow: var(--card-shadow);
        transition: var(--transition);
        position: relative;
        overflow: hidden;
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: var(--secondary-gradient);
    }

    .stat-card:hover {
        transform: translateY(-5px);
    }

    .stat-icon {
        font-size: 2.5rem;
        margin-bottom: 1rem;
        background: var(--primary-gradient);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
    }

    .stat-number {
        font-size: 2rem;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 0.5rem;
    }

    .stat-label {
        color: #718096;
        font-size: 0.9rem;
        font-weight: 500;
    }

    .chart-container {
        background: white;
        border-radius: 20px;
        padding: 2rem;
        box-shadow: var(--card-shadow);
        margin-bottom: 2rem;
    }

    .upload-form {
        background: white;
        border-radius: 15px;
        padding: 2rem;
    }

    .form-control {
        border-radius: 10px;
        border: 2px solid #e2e8f0;
        padding: 0.75rem 1rem;
        transition: var(--transition);
    }

    .form-control:focus {
        border-color: #6e8efb;
        box-shadow: 0 0 0 3px rgba(110, 142, 251, 0.1);
    }

    .btn-custom {
        background: var(--primary-gradient);
        border: none;
        border-radius: 10px;
        color: white;
        padding: 0.75rem 1.5rem;
        font-weight: 600;
        transition: var(--transition);
    }

    .btn-custom:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(110, 142, 251, 0.4);
        color: white;
    }

    .generated-links {
        margin-top: 1rem;
    }

    .link-item {
        background: #f8f9fe;
        border-radius: 10px;
        padding: 1rem;
        margin-bottom: 0.5rem;
    }

    .upload-progress {
        margin-top: 1rem;
    }

    .progress {
        height: 10px;
        border-radius: 5px;
        background: #e2e8f0;
    }

    .progress-bar {
        background: var(--primary-gradient);
        border-radius: 5px;
    }

    .date-selector {
        background: white;
        border-radius: 10px;
        padding: 1rem;
        margin-bottom: 1rem;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    }

    .date-selector select {
        border: 1px solid #e2e8f0;
        border-radius: 8px;
        padding: 0.5rem;
        margin-right: 1rem;
    }

    @media (max-width: 768px) {
        .stats-grid {
            grid-template-columns: 1fr;
        }
    }
    </style>
</head>
<body>
    <?php include 'navbar.php'; ?>

    <div class="dashboard-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h1 class="mb-2">
                        <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                    </h1>
                    <p class="mb-0 opacity-75">Gestiona tus videos y monitorea las estadísticas</p>
                </div>
                <div class="col-md-6 text-md-end">
                    <?php if (!empty($settings['counter_code'])): ?>
                        <div class="mt-3 mt-md-0"><?php echo $settings['counter_code']; ?></div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="upload-section">
            <div class="row">
                <div class="col-md-6 mb-4 mb-md-0">
                    <h4><i class="fas fa-cloud-upload-alt me-2"></i>Subir Videos</h4>
                    <form id="uploadForm" action="upload.php" method="POST" enctype="multipart/form-data" class="mt-3">
                        <div class="mb-3">
                            <label class="form-label">Seleccionar archivos</label>
                            <input type="file" name="video[]" class="form-control" accept="video/*" multiple>
                        </div>
                        <button type="submit" class="btn btn-custom">
                            <i class="fas fa-upload me-2"></i>Subir Videos
                        </button>
                        <div class="upload-progress d-none">
                            <div class="progress">
                                <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar"></div>
                            </div>
                            <small class="text-muted mt-2 d-block">Subiendo videos...</small>
                        </div>
                        <div id="uploadedLinks" class="generated-links"></div>
                    </form>
                </div>
                
                <div class="col-md-6">
                    <h4><i class="fas fa-link me-2"></i>Añadir desde URL</h4>
                    <form id="urlForm" action="add_url.php" method="POST" class="mt-3">
                        <div class="mb-3">
                            <label class="form-label">URLs de videos (.mp4)</label>
                            <textarea name="urls" class="form-control" rows="3" placeholder="Una URL por línea"></textarea>
                        </div>
                        <button type="submit" class="btn btn-custom">
                            <i class="fas fa-plus me-2"></i>Añadir URLs
                        </button>
                        <div id="urlLinks" class="generated-links"></div>
                    </form>
                </div>
            </div>
        </div>

        <div class="stats-grid">
            <div class="stat-card">
                <i class="fas fa-film stat-icon"></i>
                <div class="stat-number"><?php echo number_format($stats['total_videos']); ?></div>
                <div class="stat-label">Videos Totales</div>
            </div>
            <div class="stat-card">
                <i class="fas fa-eye stat-icon"></i>
                <div class="stat-number"><?php echo number_format($stats['total_views']); ?></div>
                <div class="stat-label">Vistas Totales</div>
            </div>
            <div class="stat-card">
                <i class="fas fa-chart-line stat-icon"></i>
                <div class="stat-number"><?php echo number_format($stats['today_views']); ?></div>
                <div class="stat-label">Vistas <?php echo date('d/m/Y'); ?></div>
            </div>
            <div class="stat-card">
                <i class="fas fa-calendar stat-icon"></i>
                <div class="stat-number"><?php echo number_format($stats['month_views']); ?></div>
                <div class="stat-label">Vistas <?php echo $nombre_mes; ?> <?php echo $selected_year; ?></div>
            </div>
        </div>

        <div class="chart-container">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h4 class="mb-0">
                    <i class="fas fa-chart-area me-2"></i>Estadísticas de Visualizaciones
                </h4>
                <div class="date-selector">
                    <form id="dateForm" class="d-flex align-items-center">
                        <select name="month" class="form-select form-select-sm me-2" onchange="this.form.submit()">
                            <?php for ($i = 1; $i <= 12; $i++): ?>
                                <option value="<?php echo $i; ?>" <?php echo $selected_month == $i ? 'selected' : ''; ?>>
                                    <?php echo $meses[sprintf("%02d", $i)]; ?>
                                </option>
                            <?php endfor; ?>
                        </select>
                        <select name="year" class="form-select form-select-sm" onchange="this.form.submit()">
                            <?php foreach ($available_years as $year): ?>
                                <option value="<?php echo $year; ?>" <?php echo $selected_year == $year ? 'selected' : ''; ?>>
                                    <?php echo $year; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>
            </div>
            <canvas id="statsChart"></canvas>
        </div>
    </div>

    <script>
    // Gráfico de estadísticas
    const ctx = document.getElementById('statsChart').getContext('2d');
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: <?php echo json_encode($all_dates); ?>,
            datasets: [{
                label: 'Visualizaciones por día',
                data: <?php echo json_encode($counts); ?>,
                borderColor: '#6e8efb',
                backgroundColor: 'rgba(110, 142, 251, 0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4,
                pointRadius: 4,
                pointBackgroundColor: '#6e8efb',
                pointHoverRadius: 6,
                pointHoverBackgroundColor: '#a777e3',
                pointHoverBorderColor: '#fff',
                pointHoverBorderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: true,
            interaction: {
                intersect: false,
                mode: 'index'
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1,
                        font: {
                            size: 12
                        }
                    },
                    grid: {
                        color: 'rgba(0, 0, 0, 0.05)'
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        font: {
                            size: 12
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    display: true,
                    position: 'top',
                    labels: {
                        font: {
                            size: 14,
                            weight: '500'
                        },
                        padding: 20
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(255, 255, 255, 0.9)',
                    titleColor: '#333',
                    titleFont: {
                        size: 14,
                        weight: 'bold'
                    },
                    bodyColor: '#666',
                    bodyFont: {
                        size: 13
                    },
                    borderColor: '#ddd',
                    borderWidth: 1,
                    padding: 12,
                    displayColors: false
                }
            }
        }
    });

    // Función para crear un elemento de enlace
    function createLinkElement(filename, originalName = '') {
        const linkItem = document.createElement('div');
        linkItem.className = 'link-item';
        const displayName = originalName || filename;
        
        linkItem.innerHTML = `
            <div class="input-group">
                <input type="text" class="form-control" value="${window.location.protocol}//${window.location.host}/${filename}" readonly>
                <button class="btn btn-outline-primary copy-btn" type="button">
                    <i class="fas fa-copy"></i>
                </button>
            </div>
            <small class="text-muted mt-1 d-block">${displayName}</small>
        `;

        // Agregar funcionalidad de copiar
        const copyBtn = linkItem.querySelector('.copy-btn');
        copyBtn.addEventListener('click', function() {
            const input = this.previousElementSibling;
            input.select();
            document.execCommand('copy');
            
            const icon = this.querySelector('i');
            const originalClass = icon.className;
            icon.className = 'fas fa-check';
            
            setTimeout(() => {
                icon.className = originalClass;
            }, 2000);
        });

        return linkItem;
    }

    // Manejo de subida de archivos
    const uploadForm = document.getElementById('uploadForm');
    const uploadProgress = uploadForm.querySelector('.upload-progress');
    const progressBar = uploadProgress.querySelector('.progress-bar');
    const uploadedLinks = document.getElementById('uploadedLinks');

    uploadForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const formData = new FormData(this);
        uploadProgress.classList.remove('d-none');
        uploadedLinks.innerHTML = '';
        
        const xhr = new XMLHttpRequest();
        xhr.open('POST', this.action, true);
        
        xhr.upload.onprogress = function(e) {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                progressBar.style.width = percentComplete + '%';
            }
        };
        
        xhr.onload = function() {
            if (xhr.status === 200) {
                const response = JSON.parse(xhr.responseText);
                if (response.success) {
                    response.files.forEach(file => {
                        uploadedLinks.appendChild(createLinkElement(file.url, file.original_name));
                    });
                }
                uploadProgress.classList.add('d-none');
                progressBar.style.width = '0%';
                uploadForm.reset();
            }
        };
        
        xhr.send(formData);
    });

    // Manejo de URLs
    const urlForm = document.getElementById('urlForm');
    const urlLinks = document.getElementById('urlLinks');

    urlForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const formData = new FormData(this);
        
        fetch(this.action, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                urlLinks.innerHTML = '';
                data.files.forEach(file => {
                    urlLinks.appendChild(createLinkElement(file.url));
                });
                this.reset();
            }
        });
    });
    </script>
</body>
</html>